/**
 * Génère une chaîne de caractères contenant une page HTML autonome complète.
 * @param {object} config - L'objet de configuration complet.
 * @returns {string|null} Le code HTML de la page, ou null si la configuration est invalide.
 */
function generateStandaloneHTML(config) {
    if (!config || !config.editorContent) {
        alert("Veuillez charger une configuration avant de générer la page HTML.");
        return null;
    }

    const content = config.editorContent;
    const videoTitle = content.interactiveVideo.video.startScreenOptions.title || "Vidéo Interactive";
    
    const videoUrl = config.context.videoUrl;
    // CORRIGÉ : La regex accepte maintenant correctement les URL "watch?v="
    const regExp = /^.*(youtu.be\/|v\/|u\/\w\/|embed\/|watch\?v=|&v=)([^#&?]*).*/;
    const match = videoUrl.match(regExp);
    const videoId = (match && match[2].length === 11) ? match[2] : null;

    if (!videoId) {
        alert("L'URL de la vidéo YouTube est invalide.");
        return null;
    }

    const cssStyles = `
        body { font-family: sans-serif; margin: 0; background-color: #f0f0f0; }
        .container { max-width: 1200px; margin: 2em auto; padding: 0 1em; }
        #preview-container { position: relative; width: 100%; }
        #youtube-player-wrapper { position: relative; width: 100%; background-color: #000; aspect-ratio: 16 / 9; border-radius: 8px; overflow: hidden; }
        #youtube-player-wrapper iframe { position: absolute; top: 0; left: 0; width: 100%; height: 100%; }
        .preview-controls { margin: 1em 0; }
        #timeline-container { position: relative; width: 100%; }
        #video-timeline { width: 100%; display: block; margin: 10px 0; }
        #timeline-markers { position: absolute; top: 50%; left: 0; width: 100%; height: 0; pointer-events: none; }
        .timeline-marker {
            position: absolute;
            width: 14px;
            height: 14px;
            background-color: #fff;
            border: 2px solid #007bff;
            border-radius: 50%;
            transform: translateY(-50%);
            cursor: pointer;
            transition: transform 0.2s ease;
            pointer-events: auto;
        }
        .timeline-marker:hover { transform: translateY(-50%) scale(1.3); }
        #question-modal-overlay { position: absolute; top: 0; left: 0; width: 100%; height: 100%; background-color: rgba(0, 0, 0, 0.75); display: flex; justify-content: center; align-items: center; z-index: 200; padding: 1em; box-sizing: border-box; }
        #question-modal-overlay.hidden { display: none; }
        #question-modal-content { background-color: #fff; padding: 2em; border-radius: 8px; width: 90%; max-width: 600px; box-shadow: 0 5px 15px rgba(0,0,0,0.3); display: flex; flex-direction: column; max-height: 90%; overflow-y: auto; }
        .answer-label { display: flex; align-items: center; padding: 0.8em; margin-bottom: 0.5em; border: 1px solid #ccc; border-radius: 5px; cursor: pointer; }
        .answer-label:hover { background-color: #f4f4f4; }
        .answer-label.correct { background-color: #d4edda; border-color: #c3e6cb; color: #155724; font-weight: bold; }
        .answer-label.incorrect { background-color: #f8d7da; border-color: #f5c6cb; color: #721c24; }
        .answered .answer-label { cursor: default; pointer-events: none; }
        #question-feedback { margin-top: 1em; padding: 1em; background-color: #e9ecef; border-left: 4px solid #007bff; border-radius: 4px; }
        #question-feedback.hidden { display: none; }
        #question-submit-btn { margin-top: 1.5em; width: 100%; padding: 12px; font-size: 16px; background-color: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; }
        #question-submit-btn:disabled { background-color: #6c757d; cursor: not-allowed; }
        #question-submit-btn:hover { background-color: #0056b3; }
    `;

    const mainLogicScript = `
        const h5pData = ${JSON.stringify(content)};
        const interactions = JSON.parse(JSON.stringify(h5pData.interactiveVideo.assets.interactions)).map(i => ({...i, triggered: false }));
        let player;
        let timelineInterval;
        let currentTimeoutId;

        function handleSeek(newTime) {
            interactions.forEach(interaction => {
                if (interaction.duration.to > newTime) {
                    interaction.triggered = false;
                }
            });
            player.seekTo(newTime, true);
        }

        function shuffleAnswers(answers) {
            const correctAnswer = answers[0];
            const shuffled = [...answers].sort(() => Math.random() - 0.5);
            const newCorrectIndex = shuffled.indexOf(correctAnswer);
            return { shuffledAnswers: shuffled, newCorrectIndex: newCorrectIndex };
        }

        function onYouTubeIframeAPIReady() {
            player = new YT.Player('youtube-player-container', {
                height: '100%', width: '100%', videoId: '${videoId}',
                playerVars: { 'playsinline': 1, 'controls': 0, 'rel': 0 },
                events: { 'onReady': onPlayerReady, 'onStateChange': onPlayerStateChange }
            });
        }
        
        function onPlayerReady(event) {
            renderTimelineMarkers();
            document.getElementById('video-timeline').addEventListener('input', (e) => {
                handleSeek(parseFloat(e.target.value));
            });
        }
        
        function onPlayerStateChange(event) {
            clearInterval(timelineInterval);
            if (event.data === YT.PlayerState.PLAYING) {
                timelineInterval = setInterval(updateTimeline, 250);
            }
        }

        function updateTimeline() {
            if (!player || typeof player.getCurrentTime !== 'function') return;
            const currentTime = player.getCurrentTime();
            const duration = player.getDuration();
            if (!duration) return;
            document.getElementById('video-timeline').value = currentTime;
            document.getElementById('video-timeline').max = duration;
            for (let i = 0; i < interactions.length; i++) {
                if (currentTime >= interactions[i].duration.to && !interactions[i].triggered) {
                    interactions[i].triggered = true;
                    player.pauseVideo();
                    showQuestionModal(interactions[i], i);
                    break;
                }
            }
        }
        
        function renderTimelineMarkers() {
            const markersContainer = document.getElementById('timeline-markers');
            const duration = player.getDuration();
            if (!duration) return;
            markersContainer.innerHTML = '';
            interactions.forEach(interaction => {
                const marker = document.createElement('div');
                marker.className = 'timeline-marker';
                const thumbWidth = 18;
                const percentage = interaction.duration.to / duration;
                marker.style.left = \`calc(\${percentage * 100}% - \${thumbWidth * percentage}px)\`;
                marker.title = \`Aller à la question (\${Math.round(interaction.duration.to)}s)\`;
                marker.addEventListener('click', (event) => {
                    event.stopPropagation();
                    handleSeek(interaction.duration.to);
                    player.playVideo();
                });
                markersContainer.appendChild(marker);
            });
        }

        function resetModal(submitHandler) {
             const answersEl = document.getElementById('question-answers');
             const feedbackEl = document.getElementById('question-feedback');
             const submitBtn = document.getElementById('question-submit-btn');
             answersEl.classList.remove('answered');
             feedbackEl.classList.add('hidden');
             submitBtn.textContent = 'Valider la réponse';
             submitBtn.disabled = false;
             answersEl.querySelectorAll('.answer-label').forEach(label => {
                label.classList.remove('correct', 'incorrect');
                label.querySelector('input').checked = false;
             });
             if (submitHandler) {
                submitBtn.onclick = submitHandler;
             }
        }

        function showQuestionModal(interaction, interactionIndex) {
            const modal = document.getElementById('question-modal-overlay');
            const questionEl = document.getElementById('question-text');
            const answersEl = document.getElementById('question-answers');
            const feedbackEl = document.getElementById('question-feedback');
            
            const behaviour = interaction.action.params.behaviour ?? {};
            const adaptivity = interaction.adaptivity ?? {};
            const autoContinue = behaviour.autoContinue ?? true;
            const enableRetry = behaviour.enableRetry ?? true;
            const enableSolutionsButton = behaviour.enableSolutionsButton ?? true;
            const timeoutCorrect = behaviour.timeoutCorrect ?? 2000;
            const timeoutWrong = behaviour.timeoutWrong ?? 3000;
            const seekBackTime = adaptivity.wrong?.seek;
            const feedbackText = interaction.action.params.overallFeedback[0].feedback;

            const continueAction = (isCorrect) => {
                modal.classList.add('hidden');
                if (!isCorrect && seekBackTime !== undefined) {
                    player.seekTo(seekBackTime, true);
                }
                player.playVideo();
            };

            const { shuffledAnswers, newCorrectIndex } = shuffleAnswers(interaction.action.params.choices[0].answers);
            
            const handleSubmit = () => {
                const submitBtn = document.getElementById('question-submit-btn');
                const selectedRadio = answersEl.querySelector('input:checked');
                if(!selectedRadio) { alert("Veuillez sélectionner une réponse."); return; }
                
                answersEl.classList.add('answered');
                const selectedIndex = parseInt(selectedRadio.value, 10);
                const isCorrect = selectedIndex === newCorrectIndex;

                if (feedbackText && feedbackText.trim() !== '') {
                    feedbackEl.innerHTML = feedbackText;
                    if(window.MathJax) MathJax.typesetPromise([feedbackEl]);
                    feedbackEl.classList.remove('hidden');
                }

                if (enableSolutionsButton) {
                    selectedRadio.parentElement.classList.add(isCorrect ? 'correct' : 'incorrect');
                    if(!isCorrect) answersEl.querySelector('input[value="' + newCorrectIndex + '"]').parentElement.classList.add('correct');
                }

                if (isCorrect) {
                    if (autoContinue) {
                        submitBtn.textContent = "Correct !";
                        submitBtn.disabled = true;
                        currentTimeoutId = setTimeout(() => continueAction(true), timeoutCorrect);
                    } else {
                        submitBtn.textContent = 'Continuer';
                        submitBtn.onclick = () => continueAction(true);
                    }
                } else {
                    if (enableRetry) {
                         if (autoContinue) {
                            submitBtn.textContent = "Incorrect...";
                            submitBtn.disabled = true;
                            currentTimeoutId = setTimeout(() => continueAction(false), timeoutWrong);
                        } else {
                            submitBtn.textContent = 'Continuer';
                            submitBtn.onclick = () => continueAction(false);
                        }
                    } else {
                        submitBtn.textContent = 'Réessayer';
                        submitBtn.onclick = () => resetModal(handleSubmit);
                    }
                }
            };

            resetModal(handleSubmit);
            questionEl.innerHTML = interaction.action.params.choices[0].question;
            answersEl.innerHTML = '';
            shuffledAnswers.forEach((ans, idx) => {
                answersEl.innerHTML += '<label class="answer-label"><input type="radio" name="preview-answer" value="' + idx + '"><span>' + ans.replace(/<\\/?p>/g, '') + '</span></label>';
            });
            
            if (window.MathJax) {
                MathJax.typesetPromise([modal]);
            }
            
            modal.classList.remove('hidden');
        }
    `;

    // --- Assemblage final de la page HTML ---
    return `
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>${videoTitle}</title>
    <style>${cssStyles}</style>
    
    <script>
      MathJax = {
        tex: {
          inlineMath: [['$', '$'], ['\\(', '\\)']],
          displayMath: [['$$', '$$'], ['\\[', '\\]']]
        }
      };
    </script>
    <script id="MathJax-script" async src="https://cdn.jsdelivr.net/npm/mathjax@3/es5/tex-mml-chtml.js"></script>

</head>
<body>
    <div class="container">
        <h1>${videoTitle}</h1>
        <div id="preview-container">
            <div id="youtube-player-wrapper">
                <div id="youtube-player-container"></div>
            </div>
            <div id="question-modal-overlay" class="hidden">
                <div id="question-modal-content">
                    <h3 id="question-text"></h3>
                    <div id="question-answers"></div>
                    <div id="question-feedback" class="hidden"></div>
                    <button id="question-submit-btn">Valider la réponse</button>
                </div>
            </div>
        </div>
        <div class="preview-controls">
            <div id="timeline-container">
                <input type="range" id="video-timeline" value="0" step="0.1">
                <div id="timeline-markers"></div>
            </div>
        </div>
    </div>
    <script src="https://www.youtube.com/iframe_api"></script>
    <script>${mainLogicScript}</script>
</body>
</html>`;
}

export function downloadStandaloneHTML(config) {
    const htmlCode = generateStandaloneHTML(config);
    if (!htmlCode) return;
    const blob = new Blob([htmlCode], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = 'video-interactive.html';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
}

export function copyStandaloneHTML(config) {
    const htmlCode = generateStandaloneHTML(config);
    if (!htmlCode) return;
    navigator.clipboard.writeText(htmlCode).then(() => {
        alert("Code HTML copié dans le presse-papiers !");
    }).catch(err => {
        console.error("Erreur de copie : ", err);
        alert("Impossible de copier le code.");
    });
}